//-----------------------------------------------------------------------------
// Copyright 2018 Thiago Alves
// This file is part of the OpenPLC Software Stack.
//
// OpenPLC is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// OpenPLC is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with OpenPLC.  If not, see <http://www.gnu.org/licenses/>.
//------
//
// This is the main file for the OpenPLC Arduino firmware. It contains the a
// Modbus RTU library to communciate with OpenPLC host as a slave device.
//
// Thiago Alves, Aug 2018
//-----------------------------------------------------------------------------

/************************PINOUT CONFIGURATION*************************

Digital In: 22, 24, 26, 28, 30, 32, 34, 36,       (%IX100.0 - %IX100.7)
            38, 40, 42, 44, 46, 48, 50, 52,       (%IX101.0 - %IX101.7)
            14, 15, 16, 17, 18, 19, 20, 21        (%IX102.0 - %IX102.7)
			
Digital Out: 23, 25, 27, 29, 31, 33, 35, 37       (%QX100.0 - %QX100.7)
             39, 41, 43, 45, 47, 49, 51, 53       (%QX101.0 - %QX101.7)

Analog In: A0, A1, A2, A3, A4, A5, A6, A7         (%IW100 - %IW107)
           A8, A9, A10, A11, A12, A13, A14, A15   (%IW108 - %IW115)
		   
Analog Out: 2, 3, 4, 5, 6, 7, 8, 9,               (%QW100 - %QW107)
            10, 11, 12, 13                        (%QW108 - %QW111)
			
*********************************************************************/

#include <Arduino.h>
#include "Modbus.h"
#include "ModbusSerial.h"

//ModBus Port information
#define BAUD        9600
#define ID          1
#define TXPIN       -1

//Define the number of registers, inputs and coils to be created
#define NUM_DISCRETE_INPUT      24
#define NUM_INPUT_REGISTERS     16
// Neu
#define NUM_COILS               16

#define NUM_HOLDING_REGISTERS   12

//Create the I/O pin masks
uint8_t pinMask_DIN[] = {22, 24, 26, 28, 30, 32, 34, 36, 38, 40, 42, 44, 46, 48, 50, 52, 14, 15, 16, 17, 18, 19, 20, 21};
uint8_t pinMask_AIN[] = {A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15};
// NEU
uint8_t pinMask_DOUT[] = {23, 25, 27, 29, 31, 33, 35, 37, 39, 41, 43, 45, 47, 49, 51, 53};
uint8_t pinMask_AOUT[] = {2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13};

//Modbus Object
ModbusSerial modbus;

/*
 * 
analogReference()



    DEFAULT: the default analog reference of 5 volts (on 5V Arduino boards) or 3.3 volts (on 3.3V Arduino boards)

    INTERNAL: an built-in reference, equal to 1.1 volts on the ATmega168 or ATmega328P and 2.56 volts on the ATmega32U4 and ATmega8 (not available on the Arduino Mega)

    INTERNAL1V1: a built-in 1.1V reference (Arduino Mega only)

    INTERNAL2V56: a built-in 2.56V reference (Arduino Mega only)

    EXTERNAL: the voltage applied to the AREF pin (0 to 5V only) is used as the reference.


 * 
 */

/*

https://etechnophiles.com/how-to-change-pwm-frequency-of-arduino-mega/
https://forum.arduino.cc/index.php?topic=72092.0


1) Arduino 2560 has 12 pins supporting PWM. They are from 2 to 13 included.

2) the PWM default frequency is 490 Hz for all pins, with the exception of pin 13 and 4,
whose frequency is 980 Hz (I checked with an oscilloscope).

3) In order to change frequency on pin 'A', we have to change some value in the timer
(or register), controlling pin 'A'. This is the list of timers in Arduino Mega 2560:

timer 0 (controls pin 13, 4);
timer 1 (controls pin 12, 11);
timer 2 (controls pin 10, 9);
timer 3 (controls pin 5, 3, 2);
timer 4 (controls pin 8, 7, 6);
 
//Code for Available PWM frequency for D2, D3 & D5:

TCCR4B = TCCR4B & B11111000 | B00000001;   // for PWM frequency of 31372.55 Hz  


//Code for Available PWM frequency for D6, D7 & D8:

TCCR4B = TCCR4B & B11111000 | B00000001;   // for PWM frequency of 31372.55 Hz  

//Code for Available PWM frequency for D11 & D12:

TCCR1B = TCCR1B & B11111000 | B00000001;  // for PWM frequency of 31372.55 Hz


//Code for Available PWM frequency for D4 & D13:

TCCR0B = TCCR0B & B11111000 | B00000001; // for PWM frequency of 62500 Hz

  
 */

void configurePins()
{
    for (int i = 0; i < NUM_DISCRETE_INPUT; i++)
    {
        pinMode(pinMask_DIN[i], INPUT);
    }
    
    for (int i = 0; i < NUM_INPUT_REGISTERS; i++)
    {
        pinMode(pinMask_AIN[i], INPUT);
    }
    
    for (int i = 0; i < NUM_COILS; i++)
    {
        pinMode(pinMask_DOUT[i], OUTPUT);
    }

    for (int i = 0; i < NUM_HOLDING_REGISTERS; i++)
    {
        pinMode(pinMask_AOUT[i], OUTPUT);
    }
    pinMode(TXPIN,OUTPUT);


//Code for Available PWM frequency for D4 & D13:

TCCR0B = TCCR0B & B11111000 | B00000001; // for PWM frequency of 62500 Hz

//Code for Available PWM frequency for D2, D3 & D5:

TCCR3B = TCCR3B & B11111000 | B00000001;   // for PWM frequency of 31372.55 Hz  

//Code for Available PWM frequency for D6, D7 & D8:

TCCR4B = TCCR4B & B11111000 | B00000001;   // for PWM frequency of 31372.55 Hz  
    
}

void setup()
{
    //Setup board I/O
    configurePins();
    analogReference(INTERNAL2V56); // 0..1023 0-2.56V
    
    //Config Modbus Serial (port, speed, rs485 tx pin)
    modbus.config(&Serial, BAUD, TXPIN);
    
    //Set the Slave ID
    modbus.setSlaveId(ID); 
    
    //Add all modbus registers
    for (int i = 0; i < NUM_DISCRETE_INPUT; ++i) 
    {
        modbus.addIsts(i);
    }
    for (int i = 0; i < NUM_INPUT_REGISTERS; ++i) 
    {
        modbus.addIreg(i);
    }
    for (int i = 0; i < NUM_COILS; ++i) 
    {
        modbus.addCoil(i);
    }
    for (int i = 0; i < NUM_HOLDING_REGISTERS; ++i) 
    {
        modbus.addHreg(i);
    }

    

    
}

void loop()
{
    //Run the main modbus task
    modbus.task();
    
    //Update modbus registers
    for (int i = 0; i < NUM_DISCRETE_INPUT; ++i) 
    {
        modbus.Ists(i, digitalRead(pinMask_DIN[i]));
    }
    for (int i = 0; i < NUM_INPUT_REGISTERS; ++i) 
    {
        // modbus.Ireg(i, (analogRead(pinMask_AIN[i]) * 64));
        modbus.Ireg(i, (analogRead(pinMask_AIN[i])));		// 0..1023 0-2.56V
		
    }
    for (int i = 0; i < NUM_COILS; ++i) 
    {
        digitalWrite(pinMask_DOUT[i], modbus.Coil(i));
    }
    for (int i = 0; i < NUM_HOLDING_REGISTERS; ++i) 
    {
        analogWrite(pinMask_AOUT[i], (modbus.Hreg(i) / 256));
    }
}
